<?php
// ===== CORS / JSON =====
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json; charset=utf-8");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') exit(0);
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error'=>'Method not allowed']);
    exit;
}

// ===== Helpers =====
function send_json($payload, $code = 200) {
    if (ob_get_length()) {
        ob_clean();
    }
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// Read JSON body
$raw = file_get_contents('php://input');
$input = json_decode($raw, true);
if (!is_array($input)) send_json(['success'=>false,'error'=>'Invalid JSON body'], 400);

// Debug log (optional)
@file_put_contents('/tmp/algo_fb_unified.log', date('[Y-m-d H:i:s] ').$raw."\n", FILE_APPEND);

// ===== AlgoLead CONFIG =====
const ALGOLEAD_API_URL = 'https://communication.algolead.org/api.php';
const ALGOLEAD_AUTH = 'cea7b36ca61d00d0677a90bc90b2d684';
const ALGOLEAD_PARTNERID = '18';
const ALGOLEAD_TRACKING = '880021';
const ALGOLEAD_FUNNEL = '600';
const ALGOLEAD_SUBID = '6687';
const ALGOLEAD_SOURCE = 'TradeGPT';

// ===== Facebook CONFIG (server-side CAPI) =====
const FB_ACCESS_TOKEN = 'EAAcDetYc184BPOs3Ez0TwwQYBmjX0HpeNwSVvtXbbjB3DC9pPnpZAjC2rpBpB4ZBGx33HPYwJpKDZA46iDFrqOYQOOz0Rop19dl0pLqCoF5lpz8eshZBsQndD1IDWScbJo9XeIOTzI8VauokGk7SRoXkMblFaovmhNPFbvMpCeZCZAavpxU7t4mbp9miIHksyhewZDZD';

// ===== IP functions (Appendix I style) =====
function cleanIP($ip) {
    $tryIP = preg_replace("/[^0-9a-f\.\,\:]/i", "", (string)$ip);
    if (strpos($tryIP, ',') !== false) {
        $ips = array_filter(array_map('trim', explode(',', $tryIP)));
        foreach ($ips as $oip) {
            if (filter_var($oip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE))
                $cleanIP = $oip;
        }
    } else $cleanIP = $tryIP;
    if (isset($cleanIP) && filter_var($cleanIP, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE))
        return $cleanIP;
    return false;
}

function getUserIP() {
    if (!isset($ip) || empty($ip)) $ip = cleanIP(@$_SERVER['HTTP_X_FORWARDED_FOR']);
    if (!isset($ip) || empty($ip)) $ip = cleanIP(@$_SERVER['HTTP_CLIENT_IP']);
    if (!isset($ip) || empty($ip)) $ip = cleanIP($_SERVER['REMOTE_ADDR'] ?? '');
    return $ip ?: false;
}

function validPublicIP($ip) {
    return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE|FILTER_FLAG_NO_RES_RANGE) ? $ip : null;
}

// ===== Country → calling code map (incl. ZA) =====
function ccFromCountry($iso2) {
    static $m = [
        'US'=>'1','CA'=>'1','ZA'=>'27','GB'=>'44','FR'=>'33','DE'=>'49','IT'=>'39','ES'=>'34','NL'=>'31','BE'=>'32',
        'CH'=>'41','AT'=>'43','DK'=>'45','SE'=>'46','NO'=>'47','FI'=>'358','PT'=>'351','GR'=>'30','PL'=>'48','CZ'=>'420',
        'HU'=>'36','RO'=>'40','BG'=>'359','HR'=>'385','SI'=>'386','SK'=>'421','EE'=>'372','LV'=>'371','LT'=>'370',
        'IE'=>'353','LU'=>'352','MT'=>'356','CY'=>'357'
    ];
    $iso2 = strtoupper((string)$iso2);
    return $m[$iso2] ?? null;
}

// ===== Phone normalization (ZA-aware) =====
function normalizePhone($raw, $countryISO) {
    $digits = preg_replace('/\D+/', '', (string)$raw);
    if ($digits === '') return [null,null,'Phone empty'];
    $cc = ccFromCountry($countryISO);
    $nat = $digits;
    // If starts with CC, strip it
    if ($cc && strpos($digits,$cc) === 0) $nat = substr($digits, strlen($cc));
    if ($cc === '27') { // South Africa
        if (isset($nat[0]) && $nat[0] === '0') $nat = substr($nat,1); // drop leading 0
        if (strlen($nat) < 9) return [null,null,'ZA national phone must be 9 digits (e.g., 821234567)'];
        return ['27', substr($nat,0,9), null];
    }
    if ($cc === '1') { // US/CA
        if (strlen($nat) == 11 && $nat[0] === '1') $nat = substr($nat,1);
        if (strlen($nat) !== 10) return [null,null,'US/CA national phone must be 10 digits'];
        return ['1', substr($nat,0,10), null];
    }
    if (!$cc) $cc = '1'; // generic fallback
    if (strlen($nat) < 7) return [null,null,'Phone too short'];
    return [$cc, $nat, null];
}

// ===== Required fields =====
foreach (['firstName','lastName','email','phone'] as $f) {
    if (empty($input[$f])) send_json(['success'=>false,'error'=>"Missing $f"], 400);
}

if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL))
    send_json(['success'=>false,'error'=>'Invalid email format'], 400);

// ===== Country (exact working logic: isset ternary, no ??) =====
$country = (isset($input['country']) && $input['country']!=='') ? strtoupper($input['country']) : 'US';

// ===== Test mode detection (skip phone validation for tests) =====
$mode = strtolower($input['mode'] ?? '');
$isTest = $mode === 'test' || 
          (strtolower($input['firstName']) === 'test' && strtolower($input['lastName']) === 'test') ||
          (isset($input['email']) && stripos($input['email'], 'test') === 0);

// Enforce test-lead rules if testing
if ($isTest) {
    $input['firstName'] = 'test';
    $input['lastName'] = 'test';
    if (stripos($input['email'], 'test') !== 0) {
        $input['email'] = 'test'.time().'@example.com';
    }
}

// ===== Phone values =====
if ($isTest) {
    if ($country === 'ZA') {
        $phonePrefix='27'; $phoneNumber='821234567';
    } // 9 digits, no leading 0
    else {
        $phonePrefix='1'; $phoneNumber='2025550199';
    } // valid US 10-digit
} else {
    list($phonePrefix, $phoneNumber, $perr) = normalizePhone($input['phone'], $country);
    if ($perr) send_json(['success'=>false,'error'=>'Phone validation','details'=>$perr], 422);
}

// ===== Client IP (allow override) =====
$clientIP = null;
if (!empty($input['clientIP'])) $clientIP = validPublicIP($input['clientIP']);
if (!$clientIP) $clientIP = getUserIP();
if (!$clientIP) $clientIP = $_SERVER['REMOTE_ADDR'] ?? '8.8.8.8';

// ===== Password =====
$password = isset($input['password']) ? (string)$input['password'] : ('AlgoLead'.rand(1000,9999));

// ===== Build AlgoLead params (exact keys/casing) =====
$algolead_params = [
    'Service' => 'CreateAccountByOptimizer',
    'PartnerID' => ALGOLEAD_PARTNERID,
    'Auth' => ALGOLEAD_AUTH,
    'TrackingID' => ALGOLEAD_TRACKING,
    'FunnelID' => ALGOLEAD_FUNNEL,
    'SubCampaignID' => ALGOLEAD_SUBID,
    'LoginEmail' => trim($input['email']),
    'LoginPassword' => $password,
    'FirstName' => trim($input['firstName']),
    'LastName' => trim($input['lastName']),
    'PhonePrefix' => preg_replace('/\D+/', '', $phonePrefix),
    'Phone' => preg_replace('/\D+/', '', $phoneNumber),
    'Country' => $country,
    'lang' => 'en-US',
    'ClientIP' => $clientIP,
    'CustomSource' => ALGOLEAD_SOURCE,
    'AffClickID' => isset($input['clickId']) ? $input['clickId'] : '',
    'AffiliatesourceID' => isset($input['sourceId']) ? $input['sourceId'] : ''
];

// Pre-flight log for quick diff with your successful curl
error_log('ALGO SEND >>> '.json_encode([
    'Service'=>$algolead_params['Service'],
    'Country'=>$algolead_params['Country'],
    'ClientIP'=>$algolead_params['ClientIP'],
    'PhonePrefix'=>$algolead_params['PhonePrefix'],
    'Phone'=>$algolead_params['Phone'],
    'isTest'=>$isTest
], JSON_UNESCAPED_SLASHES));

// ===== Call AlgoLead =====
$url = ALGOLEAD_API_URL.'?'.http_build_query($algolead_params);
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_USERAGENT => 'AlgoLead-Integration/4.0',
    CURLOPT_FOLLOWLOCATION => true
]);

$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curl_err = curl_error($ch);
curl_close($ch);

if ($curl_err) send_json(['success'=>false,'error'=>'Transport error','details'=>$curl_err], 502);

$algo = json_decode($response, true);
if (!is_array($algo))
    send_json(['success'=>false,'error'=>'Invalid response from AlgoLead','raw'=>$response,'http_code'=>$http_code], ($http_code>=400?$http_code:502));

// Normalize status
$algo_status = strtolower($algo['status'] ?? '');

// Base result
$result = [
    'success' => false,
    'algolead' => ['http'=>$http_code, 'resp'=>$algo]
];

// ===== AlgoLead SUCCESS branch =====
if ($http_code >= 200 && $http_code < 300 && $algo_status === 'success') {
    $result['success'] = true;
    $result['data'] = [
        'redirect_url' => $algo['data']['RedirectTo'] ?? null,
        'login_id' => $algo['data']['LoginID'] ?? null,
        'email' => $algo['data']['Email'] ?? $input['email'],
        'password' => $algo['data']['Password'] ?? $password
    ];

    // ---- Facebook CAPI (FIXED VERSION) ----
    $pixel_id = "726372406837726";
    
    if ($pixel_id && FB_ACCESS_TOKEN) {
        // Get cookies and fbclid
        $fbc = $input['fbc'] ?? ($_COOKIE['_fbc'] ?? null);
        $fbp = $input['fbp'] ?? ($_COOKIE['_fbp'] ?? null);
        $fbclid = $input['fbclid'] ?? null;
        
        // Build fbc from fbclid if needed
        if (!$fbc && $fbclid) {
            $fbc = 'fb.1.'.time().'.'.$fbclid;
        }
        
        // Hash functions - FIXED to return arrays like in working script
        $hashEmail = function($email) {
            if (empty($email)) return null;
            return hash('sha256', strtolower(trim($email)));
        };
        
        $hashPhone = function($phone) {
            if (empty($phone)) return null;
            // Remove non-digits
            $cleanPhone = preg_replace('/[^0-9]/', '', $phone);
            // Add US prefix if 10 digits
            if (strlen($cleanPhone) == 10) {
                $cleanPhone = '1' . $cleanPhone;
            }
            return hash('sha256', '+' . $cleanPhone);
        };
        
        // Generate event_id
        $event_id = $input['clickId'] ?? bin2hex(random_bytes(8));
        
        // Event name
        $event_name = ucfirst($input['status'] ?? 'Lead');
        
        // Build user_data - FIXED structure
        $user_data = [
            'fbc' => $fbc,
            'fbp' => $fbp,
            'client_ip_address' => $clientIP,
            'client_user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
        ];
        
        // Add hashed email as array (like in working script)
        $hashedEmail = $hashEmail($input['email']);
        if ($hashedEmail) {
            $user_data['em'] = [$hashedEmail];
        }
        
        // Add hashed phone as array (like in working script)
        $fullPhone = $phonePrefix . $phoneNumber;
        $hashedPhone = $hashPhone($fullPhone);
        if ($hashedPhone) {
            $user_data['ph'] = [$hashedPhone];
        }
        
        // Remove null values
        $user_data = array_filter($user_data, function($value) {
            return !is_null($value) && $value !== '';
        });
        
        // Build event source URL
        $event_source_url = $_SERVER['HTTP_REFERER'] ?? 
                           ((isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . 
                            ($_SERVER['HTTP_HOST'] ?? '') . ($_SERVER['REQUEST_URI'] ?? ''));
        
        // Build payload - FIXED structure
        $payload = [
            'data' => [[
                'event_name' => $event_name,
                'event_time' => time(),
                'event_id' => $event_id,
                'action_source' => 'website',
                'event_source_url' => $event_source_url,
                'user_data' => $user_data,
                'custom_data' => array_filter([
                    'currency' => $input['currency'] ?? 'USD',
                    'value' => (float)($input['payout'] ?? 0.0),
                    'click_id' => $input['clickId'] ?? null,
                    'fbclid' => $fbclid
                ]),
            ]]
        ];
        
        // Add test_event_code if provided (for Facebook Test Events)
        if (!empty($input['test_event_code'])) {
            $payload['test_event_code'] = $input['test_event_code'];
        }
        
        // Send to Facebook CAPI
        $fb_url = "https://graph.facebook.com/v20.0/{$pixel_id}/events?access_token=".FB_ACCESS_TOKEN;
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $fb_url,
            CURLOPT_POST => 1,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 20,
        ]);
        
        $fbresp = curl_exec($ch);
        $fbhttp = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $fberr = curl_error($ch);
        curl_close($ch);
        
        // Log detailed info
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'email_provided' => !empty($input['email']),
            'phone_provided' => !empty($fullPhone),
            'pixel_id' => $pixel_id,
            'event_name' => $event_name,
            'test_event_code' => $input['test_event_code'] ?? null,
            'http_code' => $fbhttp,
            'fb_response' => json_decode($fbresp, true),
            'payload_sent' => $payload
        ];
        file_put_contents('/tmp/fb_detailed_log.log', json_encode($log_entry) . "\n", FILE_APPEND);
        
        $result['facebook'] = [
            'ok' => ($fbhttp >= 200 && $fbhttp < 300) && !$fberr,
            'http' => $fbhttp,
            'error' => $fberr ?: null,
            'response' => json_decode($fbresp, true) ?: $fbresp,
            'email_sent' => !empty($input['email']),
            'phone_sent' => !empty($fullPhone),
            'test_mode' => !empty($input['test_event_code'])
        ];
    }
    
    send_json($result, 200);
}

// ===== AlgoLead FAILED branch =====
$details = $algo['errors'] ?? $algo['error'] ?? 'Registration failed';
send_json([
    'success' => false,
    'error' => $details,
    'algolead' => $result['algolead']
], 400);
?>